﻿using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using static ClickSilimulation.HotKey;

namespace ClickSilimulation
{
    public partial class Main : Form
    {
        #region 常量

        const int START_HOT_KEY_NO = 100;
        const int END_HOT_KEY_NO = 101;
        const string CONFIG_FILE_NAME = "config.json";

        #endregion

        #region 全局变量

        string[] FIRST_KEYS = EnumHelper.GetNames<KeyModifiers>();

        static List<string> invalidKeys = new List<string>()
        {
            "ShiftKey",
            "ControlKey",
            "LWin",
            "RWin",
            "Shift",
            "Control",
            "Alt",
        };
        string[] SECOND_KEYS = EnumHelper.GetNames<Keys>().Where(key => !invalidKeys.Contains(key)).ToArray();

        bool _loopFlag = false;

        ClickSilimulationConfig _config;

        ClickSilimulationConfig _defaultConfig = new ClickSilimulationConfig()
        {
            FirstKey_Start = KeyModifiers.WindowsKey,
            SecondKey_Start = Keys.F6,
            FirstKey_End = KeyModifiers.WindowsKey,
            SecondKey_End = Keys.F7,
            IntervalTime_Start = 50,
            IntervalTime_End = 200,
        };

        JsonSerializerSettings _jsonSerializerSettings = new JsonSerializerSettings()
        {
            Formatting = Formatting.Indented,
        };

        #endregion

        #region 系统事件

        public Main()
        {
            InitializeComponent();
        }

        private void Main_Load(object sender, EventArgs e)
        {
            InitConfig();
            InitControls();
            RegisterHotKey();
        }

        private void InitConfig()
        {
            if (_config == null)
            {
                LoadConfig();
            }
        }

        private void LoadConfig()
        {
            try
            {
                var strConfig = File.ReadAllText(AppDomain.CurrentDomain.BaseDirectory + CONFIG_FILE_NAME);
                var config = JsonConvert.DeserializeObject<ClickSilimulationConfig>(strConfig);
                _config = config ?? _defaultConfig;
            }
            catch (Exception)
            {
                _config = _defaultConfig;
            }
        }

        private void ConfigChanged(object sender, EventArgs e)
        {
            UpdateConfig();
        }

        /// <summary>
        /// 重载热键响应方法
        /// </summary>
        /// <param name="m"></param>
        protected override void WndProc(ref Message m)
        {
            const int WM_HOTKEY = 0x0312;
            //按快捷键 
            switch (m.Msg)
            {
                case WM_HOTKEY:
                    switch (m.WParam.ToInt32())
                    {
                        case START_HOT_KEY_NO:
                            if (!_loopFlag)
                            {
                                _loopFlag = true;
                                Task.Factory.StartNew(() =>
                                {
                                    var random = new Random();
                                    while (_loopFlag)
                                    {
                                        MouseEvent.Click();

                                        var intervalTime_Start = _config.IntervalTime_Start;
                                        var intervalTime_End = _config.IntervalTime_End < intervalTime_Start ? intervalTime_Start : _config.IntervalTime_End;
                                        Thread.Sleep(TimeSpan.FromMilliseconds(random.Next(intervalTime_Start, intervalTime_End)));
                                    }
                                });
                            }
                            break;
                        case END_HOT_KEY_NO:
                            _loopFlag = false;
                            break;
                    }
                    break;
            }
            base.WndProc(ref m);
        }

        #endregion

        #region 私有方法

        private void InitControls()
        {
            cbFK_Start.Items.AddRange(FIRST_KEYS);
            cbSK_Start.Items.AddRange(SECOND_KEYS);
            cbFK_End.Items.AddRange(FIRST_KEYS);
            cbSK_End.Items.AddRange(SECOND_KEYS);

            if (_config == null)
            {
                LoadConfig();
            }

            cbFK_Start.SelectedIndex = Array.FindIndex(FIRST_KEYS, k => k == EnumHelper.GetName(_config.FirstKey_Start));
            cbSK_Start.SelectedIndex = Array.FindIndex(SECOND_KEYS, k => k == EnumHelper.GetName(_config.SecondKey_Start));
            cbFK_End.SelectedIndex = Array.FindIndex(FIRST_KEYS, k => k == EnumHelper.GetName(_config.FirstKey_End));
            cbSK_End.SelectedIndex = Array.FindIndex(SECOND_KEYS, k => k == EnumHelper.GetName(_config.SecondKey_End));

            txtIntervalTime_Start.Text = _config.IntervalTime_Start.ToString();
            txtIntervalTime_End.Text = _config.IntervalTime_End.ToString();

            cbFK_Start.SelectedIndexChanged += ConfigChanged;
            cbSK_Start.SelectedIndexChanged += ConfigChanged;
            cbFK_End.SelectedIndexChanged += ConfigChanged;
            cbSK_End.SelectedIndexChanged += ConfigChanged;
            txtIntervalTime_Start.TextChanged += ConfigChanged;
            txtIntervalTime_End.TextChanged += ConfigChanged;
        }

        private void RegisterHotKey()
        {
            HotKey.RegisterHotKey(Handle, START_HOT_KEY_NO, _config.FirstKey_Start, _config.SecondKey_Start);
            HotKey.RegisterHotKey(Handle, END_HOT_KEY_NO, _config.FirstKey_End, _config.SecondKey_End);
        }

        private void UnregisterHotKey()
        {
            HotKey.UnregisterHotKey(Handle, START_HOT_KEY_NO);
            HotKey.UnregisterHotKey(Handle, END_HOT_KEY_NO);
        }

        private void ReregisterHotKey()
        {
            UnregisterHotKey();
            RegisterHotKey();
        }

        private void UpdateConfig()
        {
            if (_config == null)
            {
                _config = _defaultConfig;
            }

            _config.FirstKey_Start = cbFK_Start.GetKeyModifiers();
            _config.SecondKey_Start = cbSK_Start.GetKeys();
            _config.FirstKey_End = cbFK_End.GetKeyModifiers();
            _config.SecondKey_End = cbSK_End.GetKeys();
            _config.IntervalTime_Start = txtIntervalTime_Start.GetIntValue(50);
            _config.IntervalTime_End = txtIntervalTime_End.GetIntValue(200);

            if (!IsValidHotKey())
            {

            }
            SaveConfig();
            ReregisterHotKey();
        }

        private bool IsValidHotKey()
        {
            return cbFK_Start.GetKeyModifiers() != cbFK_End.GetKeyModifiers() || cbSK_Start.GetKeys() != cbSK_End.GetKeys();
        }

        private void SaveConfig()
        {
            if (_config != null)
            {
                var strConfig = JsonConvert.SerializeObject(_config, _jsonSerializerSettings);
                File.WriteAllText(AppDomain.CurrentDomain.BaseDirectory + CONFIG_FILE_NAME, strConfig);
            }
        }

        #endregion
    }
}
